/**
* \file: cmdline_parser.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Rexaline Xavier  /  RexalineInfancia.Xavier@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <errno.h>

#include "util/cfgfile_parser.h"
#include "control/configuration_defaults.h"
#include "util/helper.h"


const cfg_item_spec_t log_level_cfg_spec = {
		.kind = CFG_LOGLEVEL,
		.conf_key = "LOGLEVEL",
		.default_val.log_level = LOGGER_LEVEL_INFO};

const cfg_item_spec_t state_path_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "PERSISTENT_STATE_PATH",
		.default_val.str_val = DEFAULT_PERSISTED_STATE_PATH};

const cfg_item_spec_t script_root_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "SCRIPTS_ROOT_DIR",
		.default_val.str_val = DEFAULT_SCRIPT_ROOT_DIR};

const cfg_item_spec_t sigdb_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "SIGNATURE_DB_PATH",
		.default_val.str_val = DEFAULT_SIGNATURE_DB_PATH};

const cfg_item_spec_t sigdb_key_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "SIGNATURE_DB_PUB_KEY_PATH",
		.default_val.str_val = DEFAULT_SIGNATURE_DB_PUB_KEY_PATH};

const cfg_item_spec_t challenge_timeout_cfg_spec = {
		.kind = CFG_UINT32,
		.conf_key = "CHALLENGE_TIMEOUT",
		.default_val.u32_val = DEFAULT_CHALLENGE_TIMEOUT};

const cfg_item_spec_t script_timeout_cfg_spec = {
		.kind = CFG_UINT32,
		.conf_key = "SCRIPT_EXEC_TIMEOUT",
		.default_val.u32_val = DEFAULT_SCRIPT_EXEC_TIMEOUT};

const cfg_item_spec_t replay_trigger_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "LEVEL_REPLAY_TRIGGER_PATH",
		.default_val.str_val = DEFAULT_REPLAY_LEVEL_TRIGGER_PATH};

const cfg_item_spec_t level_change_complete_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = "LEVEL_CHANGE_COMPLETE_FILE_PATH",
		.default_val.str_val = DEFAULT_LEVEL_CHANGE_COMPLETE_FILE_PATH};

const cfg_item_spec_t challenge_verify_key_cfg_spec = {
		.kind = CFG_KEY_BEHAVIOR,
		.conf_key = "CHALLENGE_VERIFY_KEY_BEHAVIOR",
		.default_val.u32_val = DEFAULT_CHALLENGE_VERIFY_KEY_BEHAVIOR};

const cfg_item_spec_t state_val_cfg_spec = {
		.kind = CFG_UINT32,
		.conf_key = NULL,
		.default_val.u32_val = DEFAULT_ALD_LEVEL};

const cfg_item_spec_t priv_key_dest_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = DEFAULT_PRIVKEY_DEST_DIR};

const cfg_item_spec_t sigdb_privkey_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = "signature_db.priv" };

const cfg_item_spec_t sysroot_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = ""};

const cfg_item_spec_t levelnum_cfg_spec = {
		.kind = CFG_UINT32,
		.conf_key = NULL,
		.default_val.u32_val = DEFAULT_ALD_LEVEL};

const cfg_item_spec_t levelprivkey_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = DEFAULT_PRIVKEY_DEST_DIR"/level_50.priv"};

const cfg_item_spec_t serialnum_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = "local_level_change"};

const cfg_item_spec_t ecuid_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = "000000000000000"};

const cfg_item_spec_t conf_file_cfg_spec = {
		.kind = CFG_STRING,
		.conf_key = NULL,
		.default_val.str_val = DEFAULT_CONF_FILE};

const cfg_item_spec_t help_cfg_spec = {
		.kind = CFG_FLAG,
		.conf_key = NULL};


//--------------------------------------- cfgfile parser support API's -------------------------------------------------

static void cfgfile_parser_process_config_file_line(char *line_in_file, int line_no, cfg_item_t *const items[], size_t items_count)
{
	char *key=NULL;
	char *value=NULL;
	size_t idx;
	char *clean_line;
	bool is_empty=true;
	cfg_item_t *item;

	//modifies the string in line_in_conf and returns a pointer to somewhere in the line
	//the result ensures that we have no leading spaces, no trailing spaces and no newline in the string
	clean_line=helper_trim(line_in_file, &is_empty);
	//do we have a comment here or is the line empty?
	if ((clean_line[0] != '#') && !is_empty)
	{
		if (helper_extract_kv_from_line(clean_line, &key, &value, line_no) == RESULT_OK)
		{
			for (idx = 0; idx < items_count; idx++)
			{
				item = items[idx];
				if ((item != NULL) && (item->spec->conf_key != NULL) && (strcasecmp(key, item->spec->conf_key) == 0))
				{
					if (helper_item_set(item, value) != RESULT_OK)
						logger_log_error("CONFIG_PARSER - Invalid value for %s in configuration file (line %d): %s."
								" Ignoring it!", item->spec->conf_key, line_no, value);
				}
			}
		}
	}
}

// check if any config item which can be set from config file is unset
bool cfgfile_parser_is_any_cfg_item_unset(cfg_item_t *const items[], size_t items_count)
{
	size_t idx;
	cfg_item_t *item;

	for (idx = 0; idx < items_count; idx++)
	{
		item = items[idx];

		if ((!item->set) && (item->spec->conf_key != NULL))
			return true;
	}
	return false;
}

//----------------------------------------------------------------------------------------------------------------------

//--------------------------------------- global API's -----------------------------------------------------------------

error_code_t cfgfile_parser_parse_config_file(const char *config_file, cfg_item_t *const items[], size_t items_count)
{
	FILE* conf_file_fd;
	struct stat sb;
	char *line_in_file = NULL;
	int line_no = 1;
	size_t line_buf_size;

	if (stat(config_file, &sb) == -1)
	{
		if(errno == ENOENT)
			logger_log_error("CONFIG_PARSER - Config file %s not found.",config_file);

		return RESULT_INVALID_ARGS;
	}

	conf_file_fd = fopen(config_file,"r");
	if (conf_file_fd == NULL)
	{
		logger_log_error("CONFIG_PARSER - Unable to read the configuration file %s.",config_file);
		return RESULT_INVALID_ARGS;
	}

	while (getline(&line_in_file,&line_buf_size,conf_file_fd) != -1)
	{
		cfgfile_parser_process_config_file_line(line_in_file,line_no,items,items_count);
		line_no++;
	}

	if (line_in_file!=NULL)
		free(line_in_file);

	fclose(conf_file_fd);

	return RESULT_OK;
}

//-----------------------------------------------------------------------------------------------------------------------
